/*
 * routines for loading the fabric description from a file
 */
#include <stdio.h>
#include <string.h>

#include "libfma.h"
#include "lf_internal.h"
#include "lf_fabric.h"


/*
 * find a host entry in the fabric by name
 */
struct lf_host *
lf_find_host_by_name(
  struct lf_fabric *fp,
  char *hostname)
{
  int h;

  if (fp == NULL) return NULL;

  for (h=0; h<fp->num_hosts; ++h) {
    if (strcmp(hostname, fp->hosts[h]->hostname) == 0) {
      return fp->hosts[h];
    }
  }

  return NULL;
} 

/*
 * find an enclosure entry in the fabric by name
 */
struct lf_enclosure *
lf_find_enclosure_by_name(
  struct lf_fabric *fp,
  char *name)
{
  int e;

  if (fp == NULL) return NULL;

  for (e=0; e<fp->num_enclosures; ++e) {
    if (strcmp(name, fp->enclosures[e]->name) == 0) {
      return fp->enclosures[e];
    }
  }

  return NULL;
} 

/*
 * find a NIC entry by its MAC address
 */
struct lf_nic *
lf_find_nic_by_mac(
  struct lf_fabric *fp,
  lf_mac_addr_t mac)
{
  int h;

  if (fp == NULL) return NULL;

  for (h=0; h<fp->num_hosts; ++h) {
    struct lf_host *hp;
    int n;

    hp = fp->hosts[h];
    for (n=0; n<hp->num_nics; ++n) {
      struct lf_nic *np;

      np = hp->nics[n];
      if (LF_MAC_CMP(mac, np->mac_addr) == 0) {
	return np;
      }
    }
  }

  return NULL;
} 

/*
 * find a NIC entry by its MAC address
 */
struct lf_nic *
lf_find_nic_by_mac_and_host(
  struct lf_fabric *fp,
  lf_mac_addr_t mac,
  struct lf_host *hp)
{
  int h;

  if (fp == NULL) return NULL;

  for (h=0; h<fp->num_hosts; ++h) {
    struct lf_host *hp;
    int n;

    hp = fp->hosts[h];
    for (n=0; n<hp->num_nics; ++n) {
      struct lf_nic *np;

      np = hp->nics[n];
      if (LF_MAC_CMP(mac, np->mac_addr) == 0 && np->host == hp) {
	return np;
      }
    }
  }

  return NULL;
} 

/*
 * convert an internal slot number to a displayable one
 * slot numbers are 0..(num_lc_slots + num_bp_slots - 1)
 */
int
lf_slot_display_no(
  lf_linecard_t *lp)
{
  return (lp->slot < lp->enclosure->num_lc_slots) ?
          lp->slot + lp->enclosure->lc_slotbase :
          lp->slot - lp->enclosure->num_lc_slots + lp->enclosure->bp_slotbase;
}

/*
 * Find an xbar given its xbar ID
 */
struct lf_xbar *
lf_find_xbar_by_id(
  struct lf_fabric *fp,
  int xbar_id)
{
  int e;

  if (xbar_id == 0) {
    return NULL;
  }

  for (e=0; e<fp->num_enclosures; ++e) {
    struct lf_enclosure *ep;
    int l;

    ep = fp->enclosures[e];
    for (l=0; l<ep->num_slots; ++l) {
      struct lf_linecard *lp;
      int x;

      lp = ep->slots[l];
      if (lp == NULL) continue;
      for (x=0; x<lp->num_xbars; ++x) {
	struct lf_xbar *xp;

	xp = LF_XBAR(lp->xbars[x]);
	if (xp->xbar_id == xbar_id) {
	  return xp;
	}
      }
    }
  }

  return NULL;
}

/*
 * Find an xbar given its xbar ID, but not the same one as passed in
 */
struct lf_xbar *
lf_find_another_xbar_by_id(
  struct lf_fabric *fp,
  struct lf_xbar *orig_xp)
{
  int xbar_id;
  int e;

  /* get the xbar ID to look for */
  xbar_id = orig_xp->xbar_id;
  if (xbar_id == 0) {
    return NULL;
  }

  /* Find a different xbar with this ID */
  for (e=0; e<fp->num_enclosures; ++e) {
    struct lf_enclosure *ep;
    int l;

    ep = fp->enclosures[e];
    for (l=0; l<ep->num_slots; ++l) {
      struct lf_linecard *lp;
      int x;

      lp = ep->slots[l];
      if (lp == NULL) continue;
      for (x=0; x<lp->num_xbars; ++x) {
	struct lf_xbar *xp;

	xp = LF_XBAR(lp->xbars[x]);
	if (xp->xbar_id == xbar_id && xp != orig_xp) {
	  return xp;
	}
      }
    }
  }

  return NULL;
}

/*
 * Create an array of all the xbars in the fabric for easy access
 */
int
lf_build_xbar_array(
  struct lf_fabric *fp)
{
  int e;
  int xi;
  int array_size;

  array_size = fp->num_xbars;		/* it's at least this big */

  xi = 0;
  for (e=0; e<fp->num_enclosures; ++e) {
    struct lf_enclosure *ep;
    int l;

    ep = fp->enclosures[e];
    for (l=0; l<ep->num_slots; ++l) {
      struct lf_linecard *lp;
      int x;

      lp = ep->slots[l];
      if (lp == NULL) continue;
      for (x=0; x<lp->num_xbars; ++x) {

	/* make sure there is room in the xbar array */
	if (xi >= array_size) {
	  void *nxa;

	  array_size += 100;
	  nxa = realloc(fp->xbars, array_size * sizeof(struct lf_xbar *));
	  if (nxa == NULL) {
	    fp->num_xbars = 0;
	    LF_FREE(fp->xbars);
	    return -1;
	  }

	  /* got more space, update fabric variables */
	  fp->xbars = nxa;
	}

	/* save and count the xbar */
	fp->xbars[xi] = LF_XBAR(lp->xbars[x]);
	++xi;
      }
    }
  }

  /* save xbar count */
  fp->num_xbars = xi;
  return 0;
}

/*
 * Find a NIC in a host by its index
 */
struct lf_nic *
lf_find_nic_by_id(
  struct lf_host *hp,
  int nic_id)
{
  int n;

  for (n=0; n<hp->num_nics; ++n) {
    if (hp->nics[n]->host_nic_id == nic_id) {
      return hp->nics[n];
    }
  }
  return NULL;
}

/*
 * Reverse a route
 */
void
lf_reverse_route(
  unsigned char *dst,
  unsigned char *src,
  int len)
{
  src += len-1;
  while (len > 0) {
    *dst = LF_INVERT_ROUTE(*src);
    --src;
    ++dst;
    --len;
  }
}

/*
 * create a loop route
 */
int
lf_loop_route(
  unsigned char *dst,
  unsigned char *src,
  int len,
  int loop_byte)
{
  int i;

  /* create loop route */
  for (i=0; i<len; ++i) {
    dst[i] = src[i];
    dst[2*len-i] = LF_INVERT_ROUTE(src[i]);
  }
  dst[len] = loop_byte;
  return (len*2+1);
}


/*
 * Get the generic productinfo name for an N-port NIC
 */
void
lf_make_generic_nic_product_info(
  int num_ports,
  lf_string_t product_id)
{
  sprintf(product_id, "M3F-GEN%dP", num_ports);
}

/*
 * Allocate a NIC as much as possible given number of ports
 */
struct lf_nic *
lf_alloc_generic_nic(
  int num_ports)
{
  lf_string_t product_id;

  lf_make_generic_nic_product_info(num_ports, product_id);
  return lf_alloc_nic_by_product_id(product_id);
}

/*
 * Allocate a generic linecard given number of ports
 */
lf_linecard_t *
lf_allocate_generic_linecard(
  struct lf_enclosure *ep,
  int slot,
  int num_ports,
  char *serial_no)
{
  lf_string_t product_id;

  sprintf(product_id, "GENLC%dP", num_ports);
  return lf_allocate_linecard(ep, slot, product_id, serial_no);
}

/*
 * Follow a topo link
 */
union lf_node *
lf_follow_topo_link(
  union lf_node *np,
  int port,
  int *oportp)
{
  union lf_node *onp;
  int oport;

  if (np->ln_type == LF_NODE_NIC) {
    if (port >= 0 && port < LF_NIC(np)->num_ports) {
      onp = LF_NIC(np)->topo_ports[port];
      oport = LF_NIC(np)->topo_rports[port];
    } else {
      return NULL;
    }
  } else if (np->ln_type == LF_NODE_XBAR) {
    if (port >= 0 && port < LF_XBAR(np)->num_ports) {
      onp = LF_XBAR(np)->topo_ports[port];
      oport = LF_XBAR(np)->topo_rports[port];
    } else {
      return NULL;
    }
  } else {
    LF_ERROR(("Bad node type, not topo!"));
  }

  /* return the other end */
  if (oportp != NULL) *oportp = oport;
  return onp;

 except:
  return NULL;
}

/*
 * Create an ASCII description of a node
 */
char *
lf_node_string(
  union lf_node *np,
  int port)
{
  static lf_string_t buf[4];
  struct lf_linecard *lp;
  struct lf_xbar *xp;
  struct lf_nic *nicp;
  struct lf_xcvr *xcp;
  static int si;
  char *s;

  /* rotate return buf so that printf() may make more than one call */
  ++si;
  if (si >= 4) si=0;
  s = buf[si];

  switch (np->ln_type) {
    case LF_NODE_XBAR:
      xp = LF_XBAR(np);
      if (xp->linecard != NULL) {
	sprintf(s, "%s:s%d:x%d:p%d",
	    LF_XBAR(np)->linecard->enclosure->name,
	    lf_slot_display_no(LF_XBAR(np)->linecard),
	    LF_XBAR(np)->xbar_no, port);
      } else if (xp->xbar_id != 0) {
	sprintf(s, "xid %d, p%d", xp->xbar_id, port);
      } else {
	sprintf(s, "x%dp%d", xp->x_topo_index, port);
      }
      break;
    case LF_NODE_NIC:
      nicp = LF_NIC(np);
      if (nicp->host != NULL && nicp->host->hostname != NULL) {
	sprintf(s, "%s NIC %d, p%d mac=" LF_MAC_FORMAT,
	    LF_NIC(np)->host->hostname, LF_NIC(np)->host_nic_id, port,
	    LF_MAC_ARGS(LF_NIC(np)->mac_addr));
      } else {
	sprintf(s, LF_MAC_FORMAT ", p%d",
	    LF_MAC_ARGS(LF_NIC(np)->mac_addr), port);
      }
      break;
    case LF_NODE_LC_XCVR:
      xcp = LF_XCVR(np);
      lp = xcp->p.linecard;
      sprintf(s, "%s:s%d:p%d/%d", lp->enclosure->name,
	  lf_slot_display_no(lp), lp->xcvr_labels[xcp->port], port);
      break;
    case LF_NODE_NIC_XCVR:
      xcp = LF_XCVR(np);
      nicp = xcp->p.nic;
      if (nicp->host != NULL && nicp->host->hostname != NULL) {
	sprintf(s, "%s NIC %d, p%d", nicp->host->hostname,
	    nicp->host_nic_id, xcp->port);
      } else {
	sprintf(s, LF_MAC_FORMAT ", p%d", LF_MAC_ARGS(nicp->mac_addr),
	    xcp->port);
      }
      break;
      sprintf(s, "NIC XCVR not supported yet");
      break;
  }

  return s;
}

/*
 * Make a default hostname when all we know is the MAC address of a NIC
 */
void
lf_make_mac_hostname(
  char *hostname,
  lf_mac_addr_t mac_addr,
  int num_ports)
{
  sprintf(hostname, "unk:" LF_MAC_FORMAT ":%dp", LF_MAC_ARGS(mac_addr),
	  num_ports);
}

/*
 * Find the info needed to describe this end of a link in the database.
 * If this is an internal link, return NULL for name.
 * links are characterized by the physical link being the same as the topo
 */
int
lf_get_topo_link_db_info(
  union lf_node *np,
  int port,
  char **name,
  int *slot,
  int *dbport,
  int *subport)
{
  union lf_node *pnp;
  struct lf_linecard *lp;
  struct lf_xbar *xp;
  struct lf_xcvr *xcp;

  /* Get params if XBAR */
  if (np->ln_type == LF_NODE_XBAR) {

    xp = LF_XBAR(np);
    pnp = xp->phys_ports[port];

    /* If phys port == topo port, non-DB link, just return */
    if (pnp == xp->topo_ports[port]) {
      *name = NULL;

    /* Make sure this is a line-card xcvr, else we don't like it */
    } else if (pnp->ln_type != LF_NODE_LC_XCVR) {
      LF_ERROR(("Node must be LC xcvr"));

    /* Not the same, so get the connection info */
    } else {
      xcp = LF_XCVR(pnp);
      lp = xcp->p.linecard;
      *name = lp->enclosure->name;
      *slot = lf_slot_display_no(lp);
      *dbport = lp->xcvr_labels[xcp->port];
      *subport = xcp->rports[xp->phys_rports[port] - xcp->num_conns];
    }

  /* handle NIC case */
  } else if (np->ln_type == LF_NODE_NIC) {
    struct lf_nic *nicp;

    nicp = LF_NIC(np);
    pnp = nicp->phys_ports[port];

    /* If phys port == topo port, non-DB link, just return */
    if (pnp == nicp->topo_ports[port]) {
      *name = NULL;

    /* Make sure this is a NIC xcvr, else we don't like it */
    } else if (pnp->ln_type != LF_NODE_NIC_XCVR) {
      LF_ERROR(("Node must be NIC xcvr"));

    /* get the link identification for the DB */
    } else {

      *name = nicp->host->hostname;
      *slot = nicp->host_nic_id;
      *dbport = port;
      *subport = nicp->phys_rports[port] - LF_XCVR(pnp)->num_conns;
    }
  }

  return 0;

 except:
  return -1;
}

/*
 * Find the index for a given enclosure
 */
int
lf_get_enclosure_index(
  struct lf_fabric *fp,
  struct lf_enclosure *ep)
{
  int e;

  for (e=0; e<fp->num_enclosures; ++e) {
    if (fp->enclosures[e] == ep) {
      return e;
    }
  }
  return -1;
}

/*
 * Create a nice string for printing a route
 * Use several buffers so a single printf can have multiple calls to
 * this function.
 */
char *
lf_route_str(
  unsigned char *route,
  int route_len)
{
#define __NUM_ROUTE_BUFS 4
  static lf_string_t s[__NUM_ROUTE_BUFS];
  static int buf_index;
  char *o;
  char *p;
  int i;

  /* get buffer to use, and advance pointer */
  p = s[buf_index];
  o = p;
  ++buf_index;
  if (buf_index >= __NUM_ROUTE_BUFS) buf_index = 0;

  *p = '\0';
  for (i=0; i<route_len; ++i) {
    p += sprintf(p, "%d ", LF_ROUTE_TO_DELTA(route[i]));
  }
  return o;
}

/*
 * Follow a route from a node and port
 */
union lf_node *
lf_follow_route(
  union lf_node *start_np,
  int start_port,
  unsigned char *route,
  int route_len,
  int *dest_port)
{
  union lf_node *dest_np;
  int out_port;

  /* follow the first link */
  dest_np = lf_follow_topo_link(start_np, start_port, dest_port);

  /* If nothing there or no route left, we are done */
  if (dest_np == NULL || route_len <= 0) {
    return dest_np;

  /* If node is not an xbar, return nothing */
  } else if (dest_np->ln_type != LF_NODE_XBAR) {
    return NULL;

  /* xbar with more route bytes left - recursively follow! */
  } else {
    out_port = *dest_port + LF_ROUTE_TO_DELTA(route[0]);
    return lf_follow_route(dest_np, out_port, route+1, route_len-1,
				dest_port);
  }
}

/*
 * Firmware type string
 */
char *
lf_fw_type_string(
  enum lf_firmware_type fw_type)
{
  switch (fw_type) {
  case MYRI_FW_GM_1:
    return "GM-1";
  case MYRI_FW_GM_2:
    return "GM-2";
  case MYRI_FW_MX_1:
    return "MX-1";
  case MYRI_FW_XM_1:
    return "XM-1";
  case MYRI_FW_2Z_1:
    return "2Z-1";
  case MYRI_FW_LXGDB_1:
    return "LXGDB-1";
  case MYRI_FW_UNKNOWN:
    return "unknown";
    break;
  }
  return "unknown";
}
